/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::  Module      :   PROSAK Resource Framework API Header File
::  Copyright   :   (C)2002-2009 Woodward
::  Platform(s) :   MPC5xx
::  Dependants  :   MPC5xx OS Sub-project
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_Knock.h

Woodward's MPC5xx based modules provide active knock control via Motorola's PROSAK knock control integrated
circuit (IC). The complexity of the PROSAK (Programmable Stand Alone Knock Chip) is encapsulated by the
framework while the functionality of this device is abstracted to the application with the BEHAVIOUR_PROSAK_KNK
behaviour.

The PROSAK device requires an application to command a minimum of three angle domain driven control signals as
well as to read and write a plefora of internal registers inorder to provide an integrated engine detonation
detection solution. The Framework's BEHAVIOUR_PROSAK_KNK aims to provide a simple interface the PROSAK devices's
internal memory and windowing signals so that the application need not concern itself with the actual generation
of these signals or the interfacing requirements needed to access the device's memory. However an intimate
understanding of the capability and usage of the IC is assumed by this documentation. The reader is referred
to Motorola's PROSAK technical information for further information on how the PROSAK device can be employed
to implement a knock control scheme.

The BEHAVIOUR_PROSAK_KNK can be split into three main areas
- \ref prosakmemoryaccess
- \ref prosakwindowsignals
- \ref prosakevent */

#ifndef __RESOURCE_KNOCK_H
#define __RESOURCE_KNOCK_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>
#include <NativeError.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! Describes the memory map of a PROSAK device */
typedef struct
{
    uint2 R_KNKCOEFArr[20];         /* 00..13 */
    uint2 R_TSTIIRRAM_14;           /*     14 */
    uint2 R_TSTIIRRAM_15;           /*     15 */
    uint2 ProSAKReserved_16;        /*     16 */
    uint2 R_TSTIIRACCUM_17;         /*     17 */
    uint2 R_TSTIIRRAM_18;           /*     18 */
    uint2 R_TSTIIRRAM_19;           /*     19 */
    uint2 R_TSTIIRRAM_1A;           /*     1A */
    uint2 R_TSTIIRRAM_1B;           /*     1B */
    uint2 R_TSTIIRRAM_1C;           /*     1C */
    uint2 R_TSTIIRRAM_1D;           /*     1D */
    uint2 R_TSTIIRRAM_1E;           /*     1E */
    uint2 R_TSTIIRRAM_1F;           /*     1F */
    uint2 R_TSTIIRBUSLOW;           /*     20 */
    uint2 R_TSTIIRBUS;              /*     21 */
    uint2 ProSAKReserved_22_3F[30]; /* 22..3F */
    uint2 R_KNKTHRArr[12];          /* 40..4B */
    uint2 R_KNKCNU;                 /*     4C */
    uint2 R_KNKCND;                 /*     4D */
    uint2 R_KNKCKU;                 /*     4E */
    uint2 R_KNKCKD;                 /*     4F */
    uint2 R_REFA0H;                 /*     50 */
    uint2 R_REFA0L;                 /*     51 */
    uint2 R_REFA1H;                 /*     52 */
    uint2 R_REFA1L;                 /*     53 */
    uint2 R_KNKR;                   /*     54 */
    uint2 R_KNKI;                   /*     55 */
    uint2 ProSAKReserved_56_57[2];  /*  56-57 */
    uint2 R_KNKIH;                  /*     58 */
    uint2 R_KNKIL;                  /*     59 */
    uint2 R_REFIH;                  /*     5A */
    uint2 R_REFIL;                  /*     5B */
    uint2 ProSAKReserved_5C_5F[4];  /*  5C-5F */
    uint2 R_KNKSS;                  /*     60 */
    uint2 R_KNKTSTIIRHI;            /*     61 */
    uint2 R_KNKTSTIIRLO;            /*     62 */
    uint2 ProSAKReserved_63_70[14]; /*  63-70 */
    uint2 R_SETUP;                  /*     71 */
    uint2 R_TSTREG;                 /*     72 */
    uint2 R_CYLSYNC;                /*     73 */
    uint2 R_CYLSETUP;               /*     74 */
    uint2 R_CALSETUP;               /*     75 */
    uint2 R_KNKSETUP0;              /*     76 */
    uint2 R_KNKSETUP1;              /*     77 */
    uint2 R_KNKSTAT0;               /*     78 */
    uint2 R_KNKSTAT1;               /*     79 */
    uint2 ProSAKReserved_7A_7F[6];  /*  7A-7F */
    /* 16bit Writable Parameters */
    uint2 W_KNKCOEFArr[20];         /* 80..93 */
    uint2 W_TSTIIRRAM_94_9F[12];    /* 94..9F */
    uint2 W_TSTIIRBUSLOW;           /*     A0 */
    uint2 W_TSTIIRBUS;              /*     A1 */
    uint2 ProSAKReserved_A2_BF[30]; /* A2..BF */
    uint2 W_KNKTHRArr[12];          /* C0..CB */
    uint2 W_KNKCNU;                 /*     CC */
    uint2 W_KNKCND;                 /*     CD */
    uint2 W_KNKCKU;                 /*     CE */
    uint2 W_KNKCKD;                 /*     CF */
    uint2 W_REFA0H;                 /*     D0 */
    uint2 W_REFA0L;                 /*     D1 */
    uint2 W_REFA1H;                 /*     D2 */
    uint2 W_REFA1L;                 /*     D3 */
    uint2 W_KNKR;                   /*     D4 */
    uint2 W_KNKI;                   /*     D5 */
    uint2 ProSAKReserved_D6_D7[2];  /*  D6-D7 */
    uint2 W_KNKIH;                  /*     D8 */
    uint2 W_KNKIL;                  /*     D9 */
    uint2 W_REFIH;                  /*     DA */
    uint2 W_REFIL;                  /*     DB */
    uint2 ProSAKReserved_DC[4];     /*  DC-DF */
    uint2 W_KNKSS;                  /*     E0 */
    uint2 ProSAKReserved_E1_EF[15]; /* E1..EF */
    /* 8 bit writable parameters */
    uint2 W_UNLOCK;                 /*     F0 */
    uint2 W_SETUP;                  /*     F1 */
    uint2 W_TSTREG;                 /*     F2 */
    uint2 W_CYLSYNC;                /*     F3 */
    uint2 W_CYLSETUP;               /*     F4 */
    uint2 W_CALSETUP;               /*     F5 */
    uint2 W_KNKSETUP0;              /*     F6 */
    uint2 W_KNKSETUP1;              /*     F7 */
    uint2 W_KNKSTAT0;               /*     F8 */
} S_KnockDeviceMemory;

/*! Function pointer prototype. \b in_uAppDataSentOnNotify is the value of the S_KnockCreateResourceAttributes::uKnockNotifyUserData
    member when this notification event was defined during creation. See \ref prosakevent */
typedef void (*KnockCbackFuncPtrType)(NativePtrSizedInt_U in_uAppDataSentOnNotify);

/*! \brief Defines the principle modes of operation of a BEHAVIOUR_PROSAK_KNK resource. */
typedef enum
{
    KNK_STANDALONE,     /*!< Standard operation. MKIKs development signals are available for use by other behaviours */
    KNK_DEVELOPMENT     /*!< Development mode. Assumes a MKIKs knock development system is to be driven. Two development
                             signals are consumed and are not therefore available for use by other behaviours */
} E_KnockDevlopmentMode;

/*! Knock window control signal timing data structure */
typedef struct
{
    sint2 s2StartAngle; /*!< The start timing in degBfTDC with a resolution of 0.0625. Thus a 1 degBfTDC == 16.
                             A negative value represents an after firing TDC angle */
    sint2 s2StopAngle;  /*!< The stop timing in degBfTDC with a resolution of 0.0625. Thus a 1 degBfTDC == 16.
                             A negative value represents an after firing TDC angle */
} S_KnkEvent;

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid PROSAK attributes
           in the S_KnockResourceAttributes, S_KnockCreateResourceAttributes and S_KnockResourceReadAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_KnockResourceAttributes AttribsObj;

    // Going to provide values for the resource condition and the notification event condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_KNK_CONDITION | USE_KNK_CBACK_COND;
    AttribsObj.eResourceCond = RES_ENABLED;
    AttribsObj.eCbackCond = RES_ENABLED;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_KNK_CONDITION = 0x0001,             /*!< Selects S_KnockResourceAttributes::eResourceCond */
    USE_KNK_CBACK_COND = 0x0002,            /*!< Selects S_KnockResourceAttributes::eCbackCond */
    USE_KNK_KNK_WIN = 0x0004,               /*!< Selects S_KnockResourceAttributes::KnockWindowObj */
    USE_KNK_REF_WIN = 0x0008,               /*!< Selects S_KnockResourceAttributes::RefWindowObj */
    USE_KNK_SYNC_EVENT = 0x0010,            /*!< Selects S_KnockResourceAttributes::SyncEventObj */
    USE_KNK_CYL_INC = 0x0020,               /*!< Selects S_KnockResourceAttributes::CylIncObj */
    USE_KNK_CYL_EVENT = 0x0040,             /*!< Selects S_KnockResourceAttributes::CylEventObj */
    USE_KNK_MEM_PTR = 0x0080,               /*!< Selects S_KnockResourceReadAttributes::pKnkDeviceMemoryObj */
    USE_KNK_CLK_FREQ = 0x0100,              /*!< Selects S_KnockResourceReadAttributes::u4DeviceClockFreqInHz */
    USE_KNK_MODCLKF = 0x0200,               /*!< Selects S_KnockResourceReadAttributes::u1RecommendedMODCLKFSetting */
    USE_KNK_DEV_MODE = 0x0400,              /*!< Selects S_KnockCreateResourceAttributes::eDevelopmentMode */
    USE_KNK_REPORT = 0x0800,                /*!< Selects S_KnockCreateResourceAttributes::pfKnockNotifyCback */
    USE_KNK_DYNAMIC_ON_CREATE = 0x1000,     /*!< Selects S_KnockCreateResourceAttributes::DynamicObj */
    USE_KNK_CYLS_TO_OBSERVE = 0x2000,       /*!< Selects S_KnockCreateResourceAttributes::u1CylsToObserve */
    USE_KNK_TDC1_OFFSET = 0x4000,           /*!< Selects S_KnockCreateResourceAttributes::s2OffsetFromTDC1 */

/* IF THIS TYPE EXCEEDS 0x8000 THEN ALTER THE TYPE OF THE uValidAttributesMask MEMBER(S) ACCORDINGLY */

} E_KnockAttributesMask;

/*! \brief This data structure describes all of the PROSAK resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_PROSAK_KNK(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
\code
    ...
    S_KnockResourceAttributes AttribsObj;

    // Going to provide values for the resource condition and the notification event condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_KNK_CONDITION | USE_KNK_CBACK_COND;
    AttribsObj.eResourceCond = RES_ENABLED;
    AttribsObj.eCbackCond = RES_ENABLED;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_KNK_CONDITION, \ref USE_KNK_CBACK_COND, \ref USE_KNK_KNK_WIN, \ref USE_KNK_REF_WIN
    \ref USE_KNK_SYNC_EVENT, \ref USE_KNK_CYL_INC and \ref USE_KNK_CYL_EVENT ] that are valid for this this instance of
    the data structure.
    \code
    ...
    S_KnockResourceAttributes AttribsObj;

    // Going to provide values for the resource condition and the notification event condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_KNK_CONDITION | USE_KNK_CBACK_COND;
    AttribsObj.eResourceCond = RES_ENABLED;
    AttribsObj.eCbackCond = RES_ENABLED;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
    uint2 uValidAttributesMask;
/*! \brief The condition of the resource.

    The control signals of a disabled resource will remain inactive, even if encoder activity is observed. All
    behaviour reports are also blocked when a resource is disabled. Use the \ref USE_KNK_CONDITION mask to
    select this attribute. The value of this attribute upon resource creation is \ref RES_DISABLED.

    The condition is always the last attribute to be applied to the resource by the Framework. It is
    therefore possible to set the condition in the same set attribute call used to apply values to the
    other attributes described by the \ref S_KnockResourceAttributes data structure.
    
    Also see \ref prosakcontrolsignalstartup */
    E_ResourceCond eResourceCond;
/*! \brief The notification event condition.

    Notification events that are observed on the PROSAK device's "KO" pin are ignored while this attribute is
    \ref RES_DISABLED. Use the \ref USE_KNK_CBACK_COND bit mask to select this attribute. The value of this
    attribute upon resource creation is \ref RES_DISABLED. */
    E_ResourceCond eCbackCond;

/*! \brief Knock window timing object. This timing affects the PROSAK's "WINK" pin.

    Select this attribute with the \ref USE_KNK_KNK_WIN bit mask. This attribute is undefined upon resource creation
    and should be set prior to enabling the resource via the S_KnockResourceAttributes::eResourceCond member.

    See \ref prosakcylinderevent */
    S_KnkEvent KnockWindowObj;

/*! \brief Reference window timing object. This timing affects the PROSAK's "WINR" pin.

    Select this attribute with the \ref USE_KNK_REF_WIN bit mask. This attribute is undefined upon resource creation
    and should be set prior to enabling the resource via the S_KnockResourceAttributes::eResourceCond member.

    See \ref prosakcylinderevent */
    S_KnkEvent RefWindowObj;

/*! \brief Cylinder increment timing object. This timing affects the PROSAK's "CYLINC" pin.

    Select this attribute with the \ref USE_KNK_CYL_INC bit mask. This attribute is undefined upon resource creation
    and should be set prior to enabling the resource via the S_KnockResourceAttributes::eResourceCond member.

    See \ref prosakcylinderevent */
    S_KnkEvent CylIncObj;     

/*! \brief Synchronising event, this signal describes a single pulse that is fed into the MKIKs knock development
           system.

    This signal can only be affected when the S_KnockCreateResourceAttributes::eDevelopmentMode attribute is
    successfully set to \ref KNK_DEVELOPMENT during creation. Most modules do not support the necessary hardware
    to interface to the MKIKS knock development system and will fail creation if this mode is requested while
    unsupported.
    
    Select this attribute with \ref USE_KNK_SYNC_EVENT bit mask. This attribute is undefined upon resource creation
    and should be set prior to enabling the resource via the S_KnockResourceAttributes::eResourceCond member.

    See \ref prosakcylinderevent */
    S_KnkEvent SyncEventObj;

/*! \brief Cylinder synchronisation event, this signal is normally used to issue a pulse for each TDC event and
           is fed into the MKIKs knock development system.

    This signal can only be affected when the S_KnockCreateResourceAttributes::eDevelopmentMode attribute is
    successfully set to \ref KNK_DEVELOPMENT during creation. Most modules do not support the necessary hardware
    to interface to the MKIKS knock development system and will fail creation if this mode is requested while
    unsupported.
    
    Select this attribute with \ref USE_KNK_CYL_EVENT bit mask. This attribute is undefined upon resource creation
    and should be set prior to enabling the resource via the S_KnockResourceAttributes::eResourceCond member.

    See \ref prosakcylinderevent */
    S_KnkEvent CylEventObj;
} S_KnockResourceAttributes;

typedef S_KnockResourceAttributes const* S_KnockResourceAttributesPtr;

/*! \brief This data structure describes the creation attributes for a PROSAK knock resource

    Included within the structure is the \b DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_PROSAK_KNK() function and \ref USE_KNK_DYNAMIC_ON_CREATE is included in the
    S_KnockCreateResourceAttributes::uValidAttributesMask member.
    
    \code
    S_KnockCreateResourceAttributes AttribsObj;

    AttribsObj.uValidAttributesMask = USE_KNK_DEV_MODE;
    AttribsObj.eDevelopmentMode = KNK_STANDALONE;
    AttribsObj.uValidAttributesMask |= USE_KNK_CYLS_TO_OBSERVE;
    AttribsObj.u1CylsToObserve = 6;

    AttribsObj.uValidAttributesMask |= USE_KNK_REPORT;
    AttribsObj.pfKnockNotifyCback = KnockTestReportCBack;
    AttribsObj.uKnockNotifyUserData = (NativePtrSizedInt_U)in_pProSAKObj;

    AttribsObj.uValidAttributesMask |= USE_KNK_DYNAMIC_ON_CREATE;
    AttribsObj.DynamicObj.uValidAttributesMask = USE_KNK_REF_WIN | USE_KNK_CYL_INC | USE_KNK_KNK_WIN;
    AttribsObj.DynamicObj.KnockWindowObj.s2StartAngle = -3*16;
    AttribsObj.DynamicObj.KnockWindowObj.s2StopAngle = -15*16;
    AttribsObj.DynamicObj.RefWindowObj.s2StartAngle = 21*16;
    AttribsObj.DynamicObj.RefWindowObj.s2StopAngle = 15*16;
    AttribsObj.DynamicObj.CylIncObj.s2StartAngle = 33*16;
    AttribsObj.DynamicObj.CylIncObj.s2StopAngle = 27*16;
    // The rest of the data structure can remain uninitialised because the framework will ignore it

    eCreateError = CreateResourceBEHAVIOUR_PROSAK_KNK(RES_PROSAK, &AttribsObj);
    if (SUCCESS(eCreateError))
    {   ... \endcode
*/
typedef struct
{
/*! Logic-OR the attributes [\ref USE_KNK_DYNAMIC_ON_CREATE, \ref USE_KNK_REPORT, \ref USE_KNK_DEV_MODE,
    and \ref USE_KNK_CYLS_TO_OBSERVE] that are valid for this this instance of the data structure.

    \code
    S_KnockCreateResourceAttributes AttribsObj;

    AttribsObj.uValidAttributesMask = USE_KNK_DEV_MODE;
    AttribsObj.eDevelopmentMode = KNK_STANDALONE;
    AttribsObj.uValidAttributesMask |= USE_KNK_CYLS_TO_OBSERVE;
    AttribsObj.u1CylsToObserve = 6;

    AttribsObj.uValidAttributesMask |= USE_KNK_REPORT;
    AttribsObj.pfKnockNotifyCback = KnockTestReportCBack;
    AttribsObj.uKnockNotifyUserData = (NativePtrSizedInt_U)in_pProSAKObj;

    AttribsObj.uValidAttributesMask |= USE_KNK_DYNAMIC_ON_CREATE;
    AttribsObj.DynamicObj.uValidAttributesMask = USE_KNK_KNK_WIN;
    AttribsObj.DynamicObj.KnockWindowObj.s2StartAngle = -3*16;
    AttribsObj.DynamicObj.KnockWindowObj.s2StopAngle = -15*16;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */
    uint2 uValidAttributesMask;

/*! \brief Defines the notification event that the framework will execute when a PROSAK event is observed.

    See \ref prosakevent for a definition of a PROSAK event.

    Select this attribute with the \ref USE_KNK_REPORT bit mask. This attribute will default to \c NULL, disabling
    notification events if this attribute is not selected during creation. */
    KnockCbackFuncPtrType pfKnockNotifyCback;
/*! Application data that will be supplied as a parameter of the S_KnockCreateResourceAttributes::pfKnockNotifyCback
    notification callback function. Its type has been sized to allow a pointer to be stored in this parameter. */
    NativePtrSizedInt_U uKnockNotifyUserData;
/*! \brief Defines whether the resource is going to be used with an MKIKS knock development system.

    Select this attribute with the \ref USE_KNK_DEV_MODE bit mask. This attribute defaults to \ref KNK_STANDALONE
    if not selected during creation.

    Requesting \ref KNK_DEVELOPMENT arbitrarily will likely result in failure of creation as most hardware is not capable
    of supporting this feature. */
    E_KnockDevlopmentMode eDevelopmentMode;
/*! \brief The number of cylinders that this resource is performing knock monitoring for.

    It defines the number pulses that would be generated on the knock control signals that issue one pulse per cylinder
    event. See \ref prosakcylinderevent for furthe detail.
    
    Select this attribute with the \ref USE_KNK_CYLS_TO_OBSERVE bit mask. This attribute defaults to 6 if not selected. */
    uint1 u1CylsToObserve;

/*! \brief The offset from TDC1 to the first cylinder that is observed by this device in degBfTDC x 16 (1 degBfTDC = 16).

    Modules with more than one device would need to offset the second device to ensure that the cylinders would align
    correctly.

    Select this attribute with the \ref USE_KNK_TDC1_OFFSET bit mask. This attribute defaults to 0 if not selected. */
    sint2 s2OffsetFromTDC1;

/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_PROSAK_KNK(). Select with the \ref USE_KNK_DYNAMIC_ON_CREATE bit mask */
    S_KnockResourceAttributes DynamicObj;
} S_KnockCreateResourceAttributes;

typedef S_KnockCreateResourceAttributes const* S_KnockCreateResourceAttributesPtr;

/*! \brief Data structure describes all those readable BEHAVIOUR_PROSAK_KNK attributes that are not part of
           the device's virtual memory.

    These attributes can be read from the resource once it has been created using the GetResourceAttributesBEHAVIOUR_PROSAK_KNK()
    function. The \b uValidAttributesMask member is used to identify those attributes that should be read.
    \code
    S_KnockResourceReadAttributes ProSAKReadObj;

    uint2 u2W_SETUPValue;

    // Want to fill the u1RecommendedMODCLKFSetting member only
    ProSAKReadObj.uValidAttributesMask = USE_KNK_MODCLKF;
    u2W_SETUPValue = 2; // Apply a default in case the reas fails
    if (SUCCESS(GetResourceAttributesBEHAVIOUR_PROSAK_KNK(in_pProSAKObj->ResourceInfoObj.eResource, &ProSAKReadObj)))
    {
        u2W_SETUPValue = ProSAKReadObj.u1RecommendedMODCLKFSetting;
        ... \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_KNK_MEM_PTR, \ref USE_KNK_CLK_FREQ and \ref USE_KNK_MODCLKF]
    that are to be read by the framework and stored in this data structure.
    \code
    S_KnockResourceReadAttributes ProSAKReadObj;

    // Want to fill the u1RecommendedMODCLKFSetting member only
    ProSAKReadObj.uValidAttributesMask = USE_KNK_MODCLKF; \endcode */
    uint2 uValidAttributesMask;
/*! Pointer to the base of the virtual memory object that has been mapped to the device. This attribute can
    also be read directly witht he GetKnockResourceMemoryObject() function. */
    S_KnockDeviceMemory* pKnkDeviceMemoryObj;

/*! The clock frequency being fed into to the PROSAK IC. */
    uint4 u4DeviceClockFreqInHz;

/*! The recommended setting for the MODCLKF field based upon the framework's understanding of the hardware in use
    \code
    S_KnockResourceReadAttributes ProSAKReadObj;
    uint2 u2W_SETUPValue;

    ProSAKReadObj.uValidAttributesMask = USE_KNK_MODCLKF;
    u2W_SETUPValue = 2; // Apply a default in case the reas fails
    if (SUCCESS(GetResourceAttributesBEHAVIOUR_PROSAK_KNK(in_pProSAKObj->ResourceInfoObj.eResource, &ProSAKReadObj)))
    {
        u2W_SETUPValue = ProSAKReadObj.u1RecommendedMODCLKFSetting;
        ... \endcode */
    uint1 u1RecommendedMODCLKFSetting;
} S_KnockResourceReadAttributes;

typedef S_KnockResourceReadAttributes* S_KnockResourceReadAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/

/* Will recover a memory object or NULL if the object is unavailable */
/* The API functions MemReadSynch and MemWriteSynch can be used to read the Memory Mapped Knock Device */
S_KnockDeviceMemory* GetKnockResourceMemoryObject(E_ModuleResource);

/* DO NOT use these functions directly. They are implicitly called through the use of the CreateResource() */
/*   and SetResourceAttributes() macro functions                                                           */
NativeError_S CreateResourceBEHAVIOUR_PROSAK_KNK(E_ModuleResource, S_KnockCreateResourceAttributes const*);
NativeError_S DestroyResourceBEHAVIOUR_PROSAK_KNK(E_ModuleResource);
NativeError_S SetResourceAttributesBEHAVIOUR_PROSAK_KNK(E_ModuleResource, S_KnockResourceAttributes const*);
NativeError_S GetResourceAttributesBEHAVIOUR_PROSAK_KNK(E_ModuleResource, S_KnockResourceReadAttributes*);
#endif /* __RESOURCE_KNOCK_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
